% ExtractMEAData_LatticeOnly_Batch.m 
%   Version: MATLAB 2019a, ADMINISTRATIVE RIGHTS
% 	Description:
% 	
% 		Extracts data from an xlsx file and exports the data into .mat 
%       files and excel spreadsheets. All Spike trains are electrodes are
%       considered.
% 		
% 		Requires a correct file name convention. File MUST end with 
%                       Spikes.xlsx OR SPs.xlsx
% 		
% 		Performs SP grouping if necessary	
% 		
% 		.mat file OUTPUTS
% 			Spiketrains:
% 			Percent Matrix:
% 			RawCountMatrix:
% 		
% 		Excel Spreadsheet OUTPUTS
% 			Order of Electrodes:
% 			Coincidence Count Matrix:
% 		
% 	Users Setup:
% 		Folder Directory containing the xlsx file
% 		Coincidence Thresholds for Spikes and SP's
% 		SP grouping option & grouping parameters
% 		
% 	Function Requirements:
% 		[C1,C2,Deltas] = SpikeSynchrony(L1,L2,time,MaxDiff): Find
% 		coincidences between two spike trains

clear
close all
clc

%% Set-up

%Enter the name of the folder containing the batch of excel files
BatchFolder = 'Batch4Processing';

%Set the coincidence thresholds for SPs and Spikes
CoincidenceThreshold_Spikes = 0.02;     %seconds
CoincidenceThreshold_SPs = 0.1;         %seconds

%Turn SP grouping on (1) or off (0)
SPgrouping = 1;
%Set the grouping parameters if on
MaximumSeparationInGroup = 0.60;    %the maximum allowable separation (in seconds) between timestamps before they are considered to be in separate groups
MaximumGroupSize = 3;               %the maximum number of timestamps allowed in a group

%% -----------------------------------------------------------------------

%find the name of each file
CSVfiles = dir(BatchFolder);

%Remove all files which aren't in .xlsx format
ignoreFiles = [];
for i = 1:length(CSVfiles)
    
    try
        fileformat = CSVfiles(i).name(end-3:end);
        if(~strcmp(fileformat,'xlsx'))
            ignoreFiles = [ignoreFiles i];
        end
    catch
        ignoreFiles = [ignoreFiles i];
    end
    
end
CSVfiles(ignoreFiles) = [];

%Step through extraction code for each file in turn
for i = 2:length(CSVfiles)
    CSVfile = CSVfiles(i).name;
    
    disp("Extracting from " + CSVfile)
    
    %Check which spike type
    SpikeTypeIndicator = CSVfile(end-6);
    switch SpikeTypeIndicator
        case 'P'
            SpikeType = "SPs";
            MaxDifference = CoincidenceThreshold_SPs;
            disp("File is an SP record")
        case 'e'
            SpikeType = "Spikes";
            MaxDifference = CoincidenceThreshold_Spikes;
            disp("File is a spike record")
        otherwise
            disp("Wrong file name convention, spike type not determined, skipping..")
            continue
    end
    
    SpikeTrains = [];
    MinimumSpikeNum = 1;    %minimum number of spikes for consideration

    cd(BatchFolder)
    [~,TextData,AllData] = xlsread(CSVfile,1,'A1:A50000');
    [~,~,Electrodes] = xlsread(CSVfile,1,'C1:C50000');
    Index = find(TextData=="t");
    cd ..
    
    TrainNum = 1;
    maxLat = 0;

    %Grab numerical data from each electrode in the file
    for j = 1:length(Index)-1
        try
            Latencies = AllData(Index(j)+3:Index(j+1)-2);
            if(j==length(Index)-1)
                Latencies = AllData(Index(j)+3:Index(j+1)-5);
            end
            if(length(Latencies)>=MinimumSpikeNum)
                
                SpikeTrains(TrainNum).Latencies = cell2mat(Latencies);
                
                electrodeLabel = cell2mat(Electrodes(Index(j)));
                
                if(~isa(electrodeLabel,'double'))
                    
                    electrodeLabel = reverse(string(electrodeLabel));
                    
                    electrodeLabel = strrep(electrodeLabel,"A","0");
                    electrodeLabel = strrep(electrodeLabel,"B","1");
                    electrodeLabel = strrep(electrodeLabel,"C","2");
                    electrodeLabel = strrep(electrodeLabel,"D","3");
                    electrodeLabel = strrep(electrodeLabel,"E","4");
                    electrodeLabel = strrep(electrodeLabel,"F","5");
                    electrodeLabel = strrep(electrodeLabel,"G","6");
                    electrodeLabel = strrep(electrodeLabel,"H","7");
                    electrodeLabel = strrep(electrodeLabel,"I","8");
                    electrodeLabel = strrep(electrodeLabel,"K","9");
                    
                    electrodeLabel = str2double(electrodeLabel);
                    
                end
                    
                SpikeTrains(TrainNum).Electrode = electrodeLabel;
             
                if(max(cell2mat(Latencies))>maxLat)
                    maxLat = max(cell2mat(Latencies));
                end

                TrainNum = TrainNum+1;
            end
        catch
            break;
        end
    end

    if(TrainNum==1)
        SpikeTrains.Latencies = [];
        SpikeTrains.Electrode = 12;
    end
    
    fprintf('%d spiking electrodes \n',length(SpikeTrains));

    %% Perform SP grouping if appropriate
    
    if(strcmp(SpikeType,"SPs")&&(SPgrouping==1)&&~isempty(SpikeTrains(1).Latencies))
        for j = 1:length(SpikeTrains)
            Ungrouped = SpikeTrains(j).Latencies;
            Grouped = zeros(size(Ungrouped));
            Grouped(1) = Ungrouped(1);
            GroupCount = 1;
            GroupedCounter = 2;
            for k = 2:length(Ungrouped)
                if((Ungrouped(k)-Ungrouped(k-1))<MaximumSeparationInGroup)
                    if(GroupCount <= MaximumGroupSize)
                        GroupCount = GroupCount+1;
                    else
                        Grouped(GroupedCounter) = Ungrouped(k);
                        GroupedCounter = GroupedCounter+1;
                        GroupCount = 1;
                    end
                else
                    Grouped(GroupedCounter) = Ungrouped(k);
                    GroupedCounter = GroupedCounter+1;
                    GroupCount = 1;
                end
            end
            
            %Remove all the extra zeros
            Grouped = Grouped(Grouped>0);
            SpikeTrains(j).Latencies = Grouped;
        end
        
    end
    
    %% Process coincident spike statistics

    PercentMatrix = zeros(length(SpikeTrains));
    RawCountMatrix = zeros(length(SpikeTrains));

    %Create an artificial time vector
    Ts = 0.5e-4;
    time = Ts:Ts:maxLat+10;
    
    %Try to compute all coincident events, skip if error 
    try
    
        for j = 1:length(SpikeTrains)-1
            Ej = SpikeTrains(j).Electrode;
            Ejx = floor(Ej/10);
            Ejy = rem(Ej,10);
            
            for k = j+1:length(SpikeTrains)
                Ek = SpikeTrains(k).Electrode;
                Ekx = floor(Ek/10);
                Eky = rem(Ek,10);
                
                if(abs(Ekx-Ejx)<=1&&abs(Eky-Ejy)<=1)
                    t1 = SpikeTrains(j).Latencies;
                    t2 = SpikeTrains(k).Latencies;
                    L1 = round(t1/Ts);
                    L2 = round(t2/Ts);
                    [C1,C2,Deltas] = SpikeSynchrony_SampleBased(L1,L2,time,MaxDifference);
                    PercentMatrix(j,k) = 2*sum(C1)/(length(t1)+length(t2));
                    RawCountMatrix(j,k) = sum(C1);
                end  
            end
            fprintf("%d%% ",round(100*j/(length(SpikeTrains)-1)))
        end

    catch
       
        disp("Error finding coincident spikes, skipping")
        continue;
        
    end
    
    fprintf("\n")
    
 %% Compute summary stats and save
    
    
    %Save the spike trains and their computed coincidence matrices
    savefile = strcat(CSVfile(1:end-4),'mat');
    save(savefile,'SpikeTrains','PercentMatrix','RawCountMatrix');
    
%% Create an excel file with some of the data

    excelSavefile = strcat(CSVfile(1:end-5),'_ExtractedData.xlsx');
    Electrode_Number = [SpikeTrains.Electrode]; Electrode_Number = Electrode_Number(:);
    Number_of_spikes = zeros(size(Electrode_Number));
    for j = 1:length(SpikeTrains)
            Number_of_spikes(j) = length(SpikeTrains(j).Latencies);
    end
    T1 = table(Electrode_Number,Number_of_spikes);
    T2 = array2table(RawCountMatrix);
    writetable(T1,excelSavefile,'WriteVariableNames',true,'Sheet','Order of Electrodes');
    writetable(T2,excelSavefile,'WriteVariableNames',false,'Sheet','Coincidence Count Matrix');
    
end
